<?php
/**
 * Plugin Name: WooCommerce RoyatPay Gateway
 * Description: Adds RoyatPay payment gateway to WooCommerce.
 * Version: 1.0.0
 * Author: Royat.sa
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

require_once 'RoyatApplePay.php';

// Initialize the gateway
add_action('plugins_loaded', 'init_royatpay_gateway', 0);

function init_royatpay_gateway() {
    if (!class_exists('WC_Payment_Gateway')) {
        return;
    }

    class WC_Gateway_RoyatPay extends WC_Payment_Gateway {
        public function __construct() {
            $this->id                 = 'royatpay';
            $this->icon               = ''; // URL of an icon that will be displayed on checkout page
            $this->has_fields         = true;
            $this->method_title       = __('RoyatPay', 'woocommerce-royatpay-gateway');
            $this->method_description = __('Pay with your credit card via RoyatPay.', 'woocommerce-royatpay-gateway');

            // Load the settings
            $this->init_form_fields();
            $this->init_settings();

            // Define user set variables
            $this->title        = $this->get_option('title');
            $this->description  = $this->get_option('description');

            // Actions
            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('woocommerce_receipt_' . $this->id, array($this, 'receipt_page'));
            add_action('woocommerce_api_wc_gateway_' . $this->id, array($this, 'check_response'));

            // Load the text domain for translation
            add_action('plugins_loaded', array($this, 'load_textdomain'));

            // Save payment details
            $this->supports = array(
                'products'
            );

            add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));

        }

        public function load_textdomain() {
            load_plugin_textdomain('woocommerce-royatpay-gateway', false, dirname(plugin_basename(__FILE__)) . '/languages');
        }

        public function get_client_ip() {
            $ipaddress = '';
            if (getenv('HTTP_CLIENT_IP'))
                $ipaddress = getenv('HTTP_CLIENT_IP');
            else if(getenv('HTTP_X_FORWARDED_FOR'))
                $ipaddress = getenv('HTTP_X_FORWARDED_FOR');
            else if(getenv('HTTP_X_FORWARDED'))
                $ipaddress = getenv('HTTP_X_FORWARDED');
            else if(getenv('HTTP_FORWARDED_FOR'))
                $ipaddress = getenv('HTTP_FORWARDED_FOR');
            else if(getenv('HTTP_FORWARDED'))
               $ipaddress = getenv('HTTP_FORWARDED');
            else if(getenv('REMOTE_ADDR'))
                $ipaddress = getenv('REMOTE_ADDR');
            else
                $ipaddress = 'UNKNOWN';
            return $ipaddress;
        }

        public function init_form_fields() {
            $this->form_fields = array(
                'enabled' => array(
                    'title'   => __('Enable/Disable', 'woocommerce-royatpay-gateway'),
                    'type'    => 'checkbox',
                    'label'   => __('Enable RoyatPay Payment', 'woocommerce-royatpay-gateway'),
                    'default' => 'yes'
                ),
                'title' => array(
                    'title'       => __('Title', 'woocommerce-royatpay-gateway'),
                    'type'        => 'text',
                    'description' => __('This controls the title which the user sees during checkout.', 'woocommerce-royatpay-gateway'),
                    'default'     => __('Credit Card (RoyatPay)', 'woocommerce-royatpay-gateway'),
                    'desc_tip'    => true,
                ),
                'description' => array(
                    'title'       => __('Description', 'woocommerce-royatpay-gateway'),
                    'type'        => 'textarea',
                    'description' => __('This controls the description which the user sees during checkout.', 'woocommerce-royatpay-gateway'),
                    'default'     => __('Pay with your credit card via RoyatPay.', 'woocommerce-royatpay-gateway'),
                ),
                'merchant_id' => array(
                    'title'       => __('Merchant ID', 'woocommerce-royatpay-gateway'),
                    'type'        => 'text',
                    'description' => __('Your RoyatPay Merchant ID.', 'woocommerce-royatpay-gateway'),
                    'default'     => 'test',
                    'desc_tip'    => true,
                ),
                'password' => array(
                    'title'       => __('Password', 'woocommerce-royatpay-gateway'),
                    'type'        => 'password',
                    'description' => __('Your RoyatPay Password.', 'woocommerce-royatpay-gateway'),
                    'default'     => 'test',
                    'desc_tip'    => true,
                ),
                'key_file' => array(
                    'title'       => __('Key File', 'woocommerce-royatpay-gateway'),
                    'type'        => 'file',
                    'description' => __('Upload your private key file.', 'woocommerce-royatpay-gateway'),
                    'desc_tip'    => true,
                ),
                'pem_file' => array(
                    'title'       => __('PEM File', 'woocommerce-royatpay-gateway'),
                    'type'        => 'file',
                    'description' => __('Upload your PEM file.', 'woocommerce-royatpay-gateway'),
                    'desc_tip'    => true,
                ),
                'enable_apple_pay' => array(
                    'title'       => __('Enable Apple Pay', 'woocommerce-royatpay-gateway'),
                    'type'        => 'checkbox',
                    'label'       => __('Enable Apple Pay Payment', 'woocommerce-royatpay-gateway'),
                    'default'     => 'no',
                ),
                'apple_merchant_identifier' => array(
                    'title'       => __('Apple Merchant Identifier', 'woocommerce-royatpay-gateway'),
                    'type'        => 'text',
                    'description' => __('Apple merchant identifier.', 'woocommerce-royatpay-gateway'),
                    'desc_tip'    => true,
                ),
            );
        }
        
        public function process_admin_options() {
            parent::process_admin_options();
        
            // Save Key File
            if (!empty($_FILES['woocommerce_royatpay_key_file']['name'])) {
                $this->upload_file('key_file', 'woocommerce_royatpay_key_file', 'key');
            }
        
            // Save PEM File
            if (!empty($_FILES['woocommerce_royatpay_pem_file']['name'])) {
                $this->upload_file('pem_file', 'woocommerce_royatpay_pem_file', 'pem');
            }
        }
        
        private function upload_file($option_name, $file_input_name, $file_extension) {
            if (isset($_FILES[$file_input_name]) && $_FILES[$file_input_name]['error'] == UPLOAD_ERR_OK) {
                $upload_dir = plugin_dir_path(__FILE__) . 'cer/';
        
                // Ensure the directory exists
                if (!file_exists($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
        
                // Build the destination path
                $file_name = 'royatpay-' . $option_name . '.' . $file_extension;
                $destination = $upload_dir . $file_name;
        
                // Move the uploaded file to the destination
                if (move_uploaded_file($_FILES[$file_input_name]['tmp_name'], $destination)) {
                    // Save the file path to the option
                    update_option('woocommerce_royatpay_' . $option_name, $destination);
                    // wc_add_notice(__('File uploaded successfully.', 'woocommerce-royatpay-gateway'), 'success');
                } else {
                    // wc_add_notice(__('File upload failed.', 'woocommerce-royatpay-gateway'), 'error');
                }
            }
        }        
        
        
        public function payment_fields() {
            ?>

            <style>
                        #applePay {  
                            width: 100%;  
                            height: 50px;  
                            display: none;   
                            border-radius: 5px;    
                            margin-left: auto;
                            margin-right: auto;
                            margin-top: 20px;
                            background-position: 50% 50%;
                            background-color: black;
                            background-size: 60%; 
                            background-repeat: no-repeat;  
                        }
                        .royatpay-input{
                            background: #f0f4f8 !important;
                            border-radius: .375rem !important;
                            padding: .375rem .75rem !important;
                            font-size: 1rem !important;
                            font-weight: 400 !important;
                            line-height: 1.5 !important;
                            transition: border-color .15s ease-in-out, box-shadow .15s ease-in-out;
                            background-clip: padding-box;
                            border: 1px solid #dee2e6;
                        }
            </style> 
            <fieldset>
                <p class="form-row form-row-wide">
                    <div style="margin-bottom: 18px;">
                        <?php
                            if ($this->description) {
                                echo wpautop(wp_kses_post($this->description));
                            }
                        ?>
                    </div>
                    <!-- Add more images for other credit card types if needed -->
                </p>


                <div style="margin: 8px; text-align:center;">
                    <button type="button" id="applePay">
                      <svg xmlns="http://www.w3.org/2000/svg" width="50" height="50" viewBox="0 0 640 512"><path fill="#ffffff" d="M116.9 158.5c-7.5 8.9-19.5 15.9-31.5 14.9-1.5-12 4.4-24.8 11.3-32.6 7.5-9.1 20.6-15.6 31.3-16.1 1.2 12.4-3.7 24.7-11.1 33.8m10.9 17.2c-17.4-1-32.3 9.9-40.5 9.9-8.4 0-21-9.4-34.8-9.1-17.9 .3-34.5 10.4-43.6 26.5-18.8 32.3-4.9 80 13.3 106.3 8.9 13 19.5 27.3 33.5 26.8 13.3-.5 18.5-8.6 34.5-8.6 16.1 0 20.8 8.6 34.8 8.4 14.5-.3 23.6-13 32.5-26 10.1-14.8 14.3-29.1 14.5-29.9-.3-.3-28-10.9-28.3-42.9-.3-26.8 21.9-39.5 22.9-40.3-12.5-18.6-32-20.6-38.8-21.1m100.4-36.2v194.9h30.3v-66.6h41.9c38.3 0 65.1-26.3 65.1-64.3s-26.4-64-64.1-64h-73.2zm30.3 25.5h34.9c26.3 0 41.3 14 41.3 38.6s-15 38.8-41.4 38.8h-34.8V165zm162.2 170.9c19 0 36.6-9.6 44.6-24.9h.6v23.4h28v-97c0-28.1-22.5-46.3-57.1-46.3-32.1 0-55.9 18.4-56.8 43.6h27.3c2.3-12 13.4-19.9 28.6-19.9 18.5 0 28.9 8.6 28.9 24.5v10.8l-37.8 2.3c-35.1 2.1-54.1 16.5-54.1 41.5 .1 25.2 19.7 42 47.8 42zm8.2-23.1c-16.1 0-26.4-7.8-26.4-19.6 0-12.3 9.9-19.4 28.8-20.5l33.6-2.1v11c0 18.2-15.5 31.2-36 31.2zm102.5 74.6c29.5 0 43.4-11.3 55.5-45.4L640 193h-30.8l-35.6 115.1h-.6L537.4 193h-31.6L557 334.9l-2.8 8.6c-4.6 14.6-12.1 20.3-25.5 20.3-2.4 0-7-.3-8.9-.5v23.4c1.8 .4 9.3 .7 11.6 .7z"/></svg>
                    </button>
                    <p style="display: none;" id="got_notactive"></p>
                    <p style="display: none;" id="notgot">خدمة Apple Pay غير مدعومة لهذا المتصفح</p>
                    <p style="display: none;" id="success">تم الدفع بنجاح</p>
                </div>

                <div style="display: flex; align-items: center; margin: 8px 0;">
                    <hr style="flex: 1; border: 0; border-top: 1px solid #ccc;">
                    <span style="margin: 0 10px;">OR</span>
                    <hr style="flex: 1; border: 0; border-top: 1px solid #ccc;">
                </div>


                <div style="margin-top:14px;">
                    <p style="margin-top:14px;" class="form-row form-row-wide">
                        <b><label for="royatpay-card-number"><?php _e('Card Number', 'woocommerce-royatpay-gateway'); ?> <span class="required">*</span></label></b>
                        <input id="royatpay-card-number" class="royatpay-input input-text wc-credit-card-form-card-number" type="text" autocomplete="off" placeholder="•••• •••• •••• ••••" name="royatpay_card_number">
                    </p>
                    <p style="margin-top:14px;" class="form-row form-row-first">
                        <b><label for="royatpay-card-expiry"><?php _e('Expiry (MM/YY)', 'woocommerce-royatpay-gateway'); ?> <span class="required">*</span></label></b>
                        <input id="royatpay-card-expiry" class="royatpay-input input-text wc-credit-card-form-card-expiry" type="text" autocomplete="off" placeholder="MM/YY" name="royatpay_card_expiry" maxlength="5">
                    </p>
                    <p style="margin-top:14px;" class="form-row form-row-last">
                        <b><label for="royatpay-card-cvv"><?php _e('CVV', 'woocommerce-royatpay-gateway'); ?> <span class="required">*</span></label></b>
                        <input id="royatpay-card-cvv" class="royatpay-input input-text wc-credit-card-form-card-expiry" type="text" autocomplete="off" placeholder="CVV" name="royatpay_card_cvv">
                    </p>
                </div>

                <div style="display: block; margin-top: 8rem;">
                    <div style="display: block;">
                        <img src="<?php echo plugins_url( 'assets/allcards.png', __FILE__ ); ?>" alt="applePay" style="max-width: 250px; flex-shrink: 0;">
                    </div>
                </div>
        
                <div>
                    <div style="display: flex; flex-direction: column; align-items: center; justify-content: center; text-align: center; width: 100%;">
                        <a href="https://royat.sa" style="text-decoration: none;">
                            <img style="width: 100px; margin-top: 20px;" src="<?php echo plugins_url( 'assets/royat.png', __FILE__ ); ?>" alt="RoyatPay Logo">
                        </a>
                    </div>
                </div>
                
                <div class="clear"></div>
                <script src="https://applepay.cdn-apple.com/jsapi/v1.1.0/apple-pay-sdk.js"></script>
        

            </fieldset>
            <script type="text/javascript">
                var royatAppletotal = '<?php echo WC()->cart->total ?>',
                    plugin_path = '<?php echo plugin_dir_url(__FILE__) ?>',
                    apple_merchant_identifier = '<?php echo $this->get_option('apple_merchant_identifier') ?>';
              
                (function($) {
                    $(document).ready(function() {
                        $('#royatpay-card-expiry').on('input', function(e) {
                            var input = $(this).val();
                            if (input.length === 2 && !input.includes('/')) {
                                $(this).val(input + '/');
                            }
                        });
                      
   
                      
                    });
                })(jQuery);
            </script>
			<script src="<?php echo plugin_dir_url(__FILE__) . 'sz_applepay.js?v=1'; ?>"></script>
            <?php
        }        


        public function validate_fields() {

            $errors = true;

            if (!empty($_POST['royatpay_card_number']) && !empty($_POST['royatpay_card_expiry']) && !empty($_POST['royatpay_card_cvv'])) {
                $errors = false;
            }


            if(!empty($_POST['paymentToken'])) {
                $errors = false;
            }


          if($errors){            
              wc_add_notice(__('Please make sure you add payment information.', 'woocommerce-royatpay-gateway'), 'error');
              return false;
            }

            return true;
        }


       public function process_payment($order_id) {

            $order = wc_get_order($order_id);

            if(!empty($_POST['paymentToken'])) { 
              
              
              	$edfaOrderId = 'royat_apple_' . $order_id.rand(1000,9999);
              
                $paymentTokenString = stripslashes($_POST['paymentToken']);
    
                // Decode the JSON string into an associative array
                $input = json_decode($paymentTokenString);
                            
                // Concatenate values
                $concatenatedString = $input->transactionIdentifier. $edfaOrderId. $order->get_total() . 'SAR' . $this->get_option('password');

                // Reverse the concatenated string
                $reversedString = strrev($concatenatedString);

                // Convert the reversed string to uppercase
                $upperCaseString = strtoupper($reversedString);

                // Calculate the MD5 hash
                $finalHash = md5($upperCaseString);
              

                $fil = [
                        'action' => 'SALE',
                        'client_key' => $this->get_option('merchant_id'),
                        'brand' => 'applepay',
                        'order_id' => $edfaOrderId,
                        'order_amount' => $order->get_total(),
                        'order_currency' => 'SAR',
                        'order_description' => 'order #' . $order_id,
                        'payer_first_name' => $order->get_billing_first_name(),
                        'payer_last_name' => $order->get_billing_last_name(),
                        'payer_address' => 'Riyadh',
                        'payer_country' => 'SA',
                        'payer_state' => 'Riyadh',
                        'payer_city' => $order->get_billing_city(),
                        'payer_zip' => $order->get_billing_postcode(),
                        'payer_email' => $order->get_billing_email(),
                        'payer_phone' => $order->get_billing_phone(),
                        'payer_birth_date' => '1988-03-20',
                        'payer_ip' => '105.38.142.20',
                        'return_url' => 'https://www.royat.sa',
                        'hash' => $finalHash,
                        'identifier' => $input->transactionIdentifier,
                        'parameters' => json_encode($input) // JSON-encode the payment token
                 ];
              
                    $logger = wc_get_logger();

                    $context = array('source' => 'royatpay-payment');


                    $logger->info('$_POST : ' . json_encode($_POST), $context);

                    $logger->info('Fill data: ' . json_encode($fil), $context);

                    $logger->info('PT : '.$_POST['paymentToken'] , $context);
              
              
                   $response = wp_remote_post('https://api.edfapay.com/applepay/orders/s2s/sale', [
                          'body' => $fil, // Send as form data
                          'timeout' => 45,
                          'httpversion' => '1.1',
                          'blocking' => true,
                          'headers' => [], // No need to specify Content-Type for form data
                   ]);


                 if (is_wp_error($response)) {
                    wc_add_notice(__('Payment error: request not done successfully', 'woocommerce-royatpay-gateway') . $response->get_error_message(), 'error');
                    return;
                  }
                           

                // Log the response for tracking
                $logger->info('Response from payment gateway: ' . wp_remote_retrieve_body($response), $context);


                $body = json_decode(wp_remote_retrieve_body($response), true);


                if($body['status'] == 'SETTLED') {
                    $order->payment_complete();
                    $order->reduce_order_stock();
   
    
                    wc_add_notice(__('Payment successful.', 'woocommerce-royatpay-gateway'), 'success');
                    return array(
                        'result'   => 'success',
                        'redirect' => $this->get_return_url($order),
                    );
                  
                }else{
                    echo json_encode($fil);
                    wc_add_notice(__('Payment error, please try with other pay menthod', 'woocommerce-royatpay-gateway') , 'error');
                    return;
                }

                exit();
            }


            $card_number = sanitize_text_field($_POST['royatpay_card_number']);

            $expiry = explode('/',$_POST['royatpay_card_expiry']);

            $card_expiry_month = $expiry[0];

            $card_expiry_year = '20' . $expiry[1];

            $card_cvv = sanitize_text_field($_POST['royatpay_card_cvv']);


            $response = wp_remote_post('https://api.royat.sa/api/payments/create', array(
                'method' => 'POST',
                'body' => json_encode(array(
                    'email' => $order->get_billing_email(),
                    'card_number' => $card_number,
                    'card_expiry_month' => $card_expiry_month,
                    'card_expiry_year' => $card_expiry_year,
                    'card_cvv' => $card_cvv,
                    'card' => [
                        'number' => $card_number,
                        'ex_month' => $card_expiry_month,
                        'ex_year' => $card_expiry_year,
                        'cvv' => $card_cvv
                    ],

                    'customer' => [
                        'name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                        'email' => $order->get_billing_email(),
                        'address' => $order->get_billing_address_1(),
                        'zip' => $order->get_billing_postcode(),
                        'city' => $order->get_billing_city(),
                        'ip' => $this->get_client_ip(),
                        'redirect' => get_site_url() . '/wc-api/wc_gateway_royatpay',
                    ],

                    'order' => [
                        'id' => $order_id,
                        'description' => 'order #' . $order->get_order_number(),
                        'amount' => $order->get_total(),
                    ],

                    'user' => [
                        'key' => $this->get_option('merchant_id'),
                        'password' => $this->get_option('password'),
                    ]
                )),
                'headers' => array(
                    'Content-Type' => 'application/json',
                ),
            ));

            if (is_wp_error($response)) {
                wc_add_notice(__('Payment error: ', 'woocommerce-royatpay-gateway') . $response->get_error_message(), 'error');
                return;
            }

            $body = json_decode(wp_remote_retrieve_body($response), true);

            if (isset($body['result']) && strtolower($body['result']) === 'success') {
                $order->payment_complete();
                $order->reduce_order_stock();

                WC()->session->set('royatpay_order_id',$order_id);
                WC()->session->set('royatpay_card_number', $card_number);
                WC()->session->set('royatpay_email', $order->get_billing_email());
                WC()->session->set('royatpay_trans_id', $body['trans_id']);

                wc_add_notice(__('Payment successful.', 'woocommerce-royatpay-gateway'), 'success');
                return array(
                    'result'   => 'success',
                    'redirect' => $this->get_return_url($order),
                );

            } elseif (isset($body['result']) && strtolower($body['result']) === 'redirect') {
                WC()->session->set('royatpay_order_id',$order_id);
                WC()->session->set('royatpay_card_number', $card_number);
                WC()->session->set('royatpay_email', $order->get_billing_email());
                WC()->session->set('royatpay_trans_id', $body['trans_id']);

                return array(
                    'result'   => 'success',
                    'redirect' => $body['url']
                );

            } else {

                wc_add_notice(__('Payment failed: ', 'woocommerce-royatpay-gateway') . $body['message'], 'error');
                return;

            }
        }

        public function statusTrans() {

            function generateHash($email, $password, $trans_id, $card_number) {
                // Reverse the email
                $reversed_email = strrev($email);
                // Get the first 6 characters of card_number
                $first_part = substr($card_number, 0, 6);
                // Get the last 4 characters of card_number
                $last_part = substr($card_number, -4);
                // Concatenate and reverse the result
                $reversed_card_part = strrev($first_part . $last_part);
                // Concatenate reversed email, PASSWORD, trans_id, and reversed card part
                $concatenated_string = $reversed_email . $password . $trans_id . $reversed_card_part;
                // Convert the concatenated string to uppercase
                $uppercase_string = strtoupper($concatenated_string);
                // Generate MD5 hash
                $hash = md5($uppercase_string);
                
                return $hash;
            }

            $response = wp_remote_post('https://api.edfapay.com/payment/post', array(
                'method' => 'POST',
                'body' => [
                    'action' => 'GET_TRANS_STATUS',
                    'client_key' => $this->get_option('merchant_id'),
                    'trans_id' => WC()->session->get('royatpay_trans_id'),
                    'hash' => generateHash(
                        WC()->session->get('royatpay_email'),
                        $this->get_option('password'),
                        WC()->session->get('royatpay_trans_id'),
                        WC()->session->get('royatpay_card_number')
                    )
                ],
                'headers' => array(
                    'Content-Type' => 'application/x-www-form-urlencoded',
                ),
            ));
            
            if (is_wp_error($response)) {
                wc_add_notice(__('Payment error: ', 'woocommerce-royatpay-gateway') . $response->get_error_message(), 'error');
                return;
            }            

            $body = json_decode(wp_remote_retrieve_body($response), true);

            return $body;

        }

        public function check_response() {

            if (!empty(WC()->session->get('royatpay_trans_id')) && !empty(WC()->session->get('royatpay_order_id'))) {

                $data = $this->statusTrans();

                $order_id = WC()->session->get('royatpay_order_id');
                $order = wc_get_order($order_id);

                if ($data['status'] !== 'DECLINED') {
                    $order->payment_complete();
                    $order->reduce_order_stock();
                    wc_add_notice(__('Payment successful.', 'woocommerce-royatpay-gateway'), 'success');
                    wp_redirect($this->get_return_url($order));
                    exit;
                } else {
                    wc_add_notice(__('Payment failed or was cancelled.', 'woocommerce-royatpay-gateway'), 'error');
                    wp_redirect(wc_get_checkout_url());
                    exit;
                }
            }else{
                wc_add_notice(__('Please contact to support to help you !.', 'woocommerce-royatpay-gateway'), 'error');
                wp_redirect(wc_get_checkout_url());
                exit;
            }
        }
      

        public function enqueue_scripts() {
            // Only enqueue Apple Pay script if Apple Pay is enabled
                $script_url = plugin_dir_url(__FILE__) . 'sz_applepay.js';
               // wp_enqueue_script('royatpay-apple-pay', $script_url, array(), '1.1.1', true);
                
                wp_localize_script('royatpay-apple-pay', 'applepay_object', array(
                    'total' => WC()->cart->total,
                    'plugin_path' => plugin_dir_url(__FILE__),
                    'apple_merchant_identifier' => $this->get_option('apple_merchant_identifier')
                ));
        } 
      

    }

    function add_royatpay_gateway($methods) {
        $methods[] = 'WC_Gateway_RoyatPay';
        return $methods;
    }

    add_filter('woocommerce_payment_gateways', 'add_royatpay_gateway');
}
?>